﻿Imports System.IO
Imports System.Drawing

Public Class HighScoresForm
    Private filePath As String
    Private allScores As New List(Of (Player As String, Score As Integer, Time As Integer, DateStr As String, Difficulty As String))

    Private dgv As DataGridView
    Private highlightPlayer As String
    Private highlightScore As Integer
    Private lblRank As Label
    Private glowTimer As Timer
    Private glowStep As Integer = 0
    Private glowIncreasing As Boolean = True
    Private glowTimeElapsed As Integer = 0

    Private uiFont As New Font("Segoe UI", 12, FontStyle.Regular)
    Private uiLabelFont As New Font("Segoe UI", 13, FontStyle.Bold)

    Private cmbFilter As ComboBox

    Public Sub New(path As String, Optional player As String = "", Optional score As Integer = -1)
        InitializeComponent()
        filePath = path
        highlightPlayer = player
        highlightScore = score

        ' --- Form settings ---
        Me.Font = uiFont
        Me.Text = "🏆 High Scores"
        Me.Size = New Size(820, 546)
        Me.StartPosition = FormStartPosition.CenterParent
        Me.BackColor = Color.Bisque
        Me.FormBorderStyle = FormBorderStyle.FixedDialog
        Me.MaximizeBox = False
        Me.MinimizeBox = False

        lblRank = New Label() With {
        .Font = uiLabelFont,
    .Dock = DockStyle.Top,
    .Height = 40,
    .TextAlign = ContentAlignment.MiddleCenter,
    .ForeColor = Color.DarkSlateBlue,
    .Visible = False
}
        Controls.Add(lblRank)

        ' --- Glow animation timer ---
        glowTimer = New Timer() With {.Interval = 80}
        AddHandler glowTimer.Tick, AddressOf AnimateGlow

        dgv = New DataGridView() With {
        .Font = uiFont,
        .ColumnHeadersHeight = 28,
    .Dock = DockStyle.Top,
    .Height = 270,
    .ReadOnly = True,
    .AllowUserToAddRows = False,
    .AllowUserToDeleteRows = False,
    .AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill,
    .BackgroundColor = Color.White,
    .RowTemplate = New DataGridViewRow() With {.Height = 28}
}

        ' --- Difficulty filter ---
        Dim filterPanel As New FlowLayoutPanel() With {
    .Dock = DockStyle.Top,
    .Height = 40,
    .Padding = New Padding(10, 5, 10, 5),
    .FlowDirection = FlowDirection.LeftToRight
}

        Dim lblFilter As New Label() With {
    .Text = "Filter by Difficulty:",
    .Font = uiFont,
    .AutoSize = True,
    .TextAlign = ContentAlignment.MiddleLeft,
    .Margin = New Padding(0, 5, 5, 0)
}

        cmbFilter = New ComboBox() With {
    .Font = uiFont,
    .DropDownStyle = ComboBoxStyle.DropDownList,
    .Width = 150
}
        cmbFilter.Items.AddRange(New String() {"All", "Easy", "Normal", "Hard", "Expert"})
        cmbFilter.SelectedIndex = 0 ' Default = All
        AddHandler cmbFilter.SelectedIndexChanged, AddressOf FilterChanged

        filterPanel.Controls.Add(lblFilter)
        filterPanel.Controls.Add(cmbFilter)
        Controls.Add(filterPanel)

        dgv.Columns.Add("Rank", "#")
        dgv.Columns.Add("Player", "Player")
        dgv.Columns.Add("Score", "Score")
        dgv.Columns.Add("Time", "Time (s)")
        dgv.Columns.Add("Date", "Date")
        dgv.Columns.Add("Difficulty", "Difficulty")
        Controls.Add(dgv)

        ' --- Buttons panel ---
        Dim panel As New FlowLayoutPanel() With {
            .Dock = DockStyle.Bottom,
            .Height = 55,
            .FlowDirection = FlowDirection.RightToLeft,
            .Padding = New Padding(10)
        }

        Dim btnClose As New Button() With {.Text = "Close", .Width = 80, .AutoSize = True}
        AddHandler btnClose.Click, Sub() Me.Close()

        Dim btnClear As New Button() With {.Text = "Clear Scores", .Width = 100, .AutoSize = True}
        AddHandler btnClear.Click, AddressOf ClearScores

        panel.Controls.Add(btnClose)
        panel.Controls.Add(btnClear)
        Controls.Add(panel)

        ' --- Load scores when form opens ---
        LoadScores()
    End Sub

    Private Sub LoadScores()
        dgv.Rows.Clear()
        allScores.Clear()

        If Not File.Exists(filePath) Then
            MessageBox.Show("No high scores yet!", "Leaderboard")
            Return
        End If

        Dim scores = File.ReadAllLines(filePath)
        For Each line In scores
            Dim parts = line.Split(","c)
            If parts.Length >= 5 Then
                Dim s As Integer
                Dim t As Integer
                Integer.TryParse(parts(1), s)
                Integer.TryParse(parts(2), t)
                allScores.Add((parts(0), s, t, parts(3), parts(4)))
            ElseIf parts.Length = 4 Then
                ' Backward compatibility
                Dim s As Integer
                Dim t As Integer
                Integer.TryParse(parts(1), s)
                Integer.TryParse(parts(2), t)
                allScores.Add((parts(0), s, t, parts(3), "Normal"))
            End If
        Next

        ' Load into grid
        ApplyFilter()

        ''Dim top = scoreList.OrderByDescending(Function(x) x.Score).ToList()
        ''Dim playerRank As Integer = 0
        ''Dim totalPlayers As Integer = top.Count

        ''For i As Integer = 0 To Math.Min(top.Count - 1, 19)
        ''Dim rank As Integer = i + 1

        ' Add medal icons for top 3
        ''Dim displayName As String = top(i).Player
        ''Select Case i
        ''Case 0 : displayName = "🥇 " & displayName
        ''Case 1 : displayName = "🥈 " & displayName
        ''Case 2 : displayName = "🥉 " & displayName
        ''End Select

        ''Dim idx = dgv.Rows.Add(rank.ToString(), displayName, top(i).Score, top(i).Time, top(i).DateStr, top(i).Difficulty)
        ''Dim row = dgv.Rows(idx)

        ' Row colors
        ''Select Case i
        ''Case 0 : row.DefaultCellStyle.BackColor = Color.Gold
        ''Case 1 : row.DefaultCellStyle.BackColor = Color.Silver
        ''Case 2 : row.DefaultCellStyle.BackColor = Color.FromArgb(205, 127, 50) ' Bronze
        ''Case Else : row.DefaultCellStyle.BackColor = Color.Beige
        ''End Select

        ' Highlight current player
        ''If String.Equals(top(i).Player, highlightPlayer, StringComparison.OrdinalIgnoreCase) AndAlso top(i).Score = highlightScore Then
        ''row.DefaultCellStyle.BackColor = Color.LightGreen
        ''row.DefaultCellStyle.Font = New Font(dgv.Font, FontStyle.Bold)
        ''dgv.FirstDisplayedScrollingRowIndex = idx
        ''playerRank = rank
        ''End If
        ''Next

        ' --- Show animated rank label if applicable ---
        ''If playerRank > 0 Then
        ''lblRank.Text = $"🏅 You ranked #{playerRank} out of {totalPlayers} players!"
        ''lblRank.Visible = True
        ''glowStep = 0
        ''glowIncreasing = True
        ''glowTimeElapsed = 0
        ''glowTimer.Start()
        ''End If
    End Sub

    Private Sub FilterChanged(sender As Object, e As EventArgs)
        ApplyFilter()
    End Sub

    Private Sub ApplyFilter()
        dgv.Rows.Clear()
        If allScores.Count = 0 Then Return

        Dim selectedDiff As String = cmbFilter.SelectedItem.ToString()
        Dim filtered = allScores

        If selectedDiff <> "All" Then
            filtered = allScores.Where(Function(x) x.Difficulty.Equals(selectedDiff, StringComparison.OrdinalIgnoreCase)).ToList()
        End If

        Dim top = filtered.OrderByDescending(Function(x) x.Score).ToList()
        Dim playerRank As Integer = 0
        Dim totalPlayers As Integer = top.Count

        For i As Integer = 0 To Math.Min(top.Count - 1, 19)
            Dim rank As Integer = i + 1
            Dim displayName As String = top(i).Player
            Select Case i
                Case 0 : displayName = "🥇 " & displayName
                Case 1 : displayName = "🥈 " & displayName
                Case 2 : displayName = "🥉 " & displayName
            End Select

            Dim idx = dgv.Rows.Add(rank.ToString(), displayName, top(i).Score, top(i).Time, top(i).DateStr, top(i).Difficulty)
            Dim row = dgv.Rows(idx)

            Select Case i
                Case 0 : row.DefaultCellStyle.BackColor = Color.Gold
                Case 1 : row.DefaultCellStyle.BackColor = Color.Silver
                Case 2 : row.DefaultCellStyle.BackColor = Color.FromArgb(205, 127, 50)
                Case Else : row.DefaultCellStyle.BackColor = Color.Beige
            End Select

            ' Highlight the current player's row (if applicable)
            If String.Equals(top(i).Player, highlightPlayer, StringComparison.OrdinalIgnoreCase) AndAlso top(i).Score = highlightScore Then
                row.DefaultCellStyle.BackColor = Color.LightGreen
                row.DefaultCellStyle.Font = New Font(dgv.Font, FontStyle.Bold)
                dgv.FirstDisplayedScrollingRowIndex = idx
                playerRank = rank
            End If
        Next

        ' Update rank label if player exists in this filtered view
        If playerRank > 0 Then
            lblRank.Text = $"🏅 You ranked #{playerRank} out of {totalPlayers} ({cmbFilter.SelectedItem}) players!"
            lblRank.Visible = True
            glowStep = 0
            glowIncreasing = True
            glowTimeElapsed = 0
            glowTimer.Start()
        Else
            lblRank.Visible = False
            glowTimer.Stop()
        End If
    End Sub

    ' --- Label glow animation ---
    Private Sub AnimateGlow(sender As Object, e As EventArgs)
        ' Increase elapsed time (80ms per tick)
        glowTimeElapsed += glowTimer.Interval

        ' Stop after 5 seconds (5000 ms)
        If glowTimeElapsed >= 5000 Then
            glowTimer.Stop()
            lblRank.ForeColor = Color.DarkSlateBlue ' Return to solid color
            Exit Sub
        End If

        ' Pulse effect
        If glowIncreasing Then
            glowStep += 15
            If glowStep >= 255 Then
                glowStep = 255
                glowIncreasing = False
            End If
        Else
            glowStep -= 15
            If glowStep <= 100 Then
                glowStep = 100
                glowIncreasing = True
            End If
        End If

        lblRank.ForeColor = Color.FromArgb(glowStep, 72, 61, 139) ' Animate color intensity
    End Sub

    Private Sub ClearScores(sender As Object, e As EventArgs)
        If Not File.Exists(filePath) Then
            MessageBox.Show("No scores to clear.", "Clear Scores")
            Return
        End If

        Dim confirm = MessageBox.Show("Are you sure you want to clear all high scores?", "Confirm Clear", MessageBoxButtons.YesNo, MessageBoxIcon.Warning)
        If confirm = DialogResult.Yes Then
            Try
                File.Delete(filePath)
                dgv.Rows.Clear()
                lblRank.Visible = False
                glowTimer.Stop()
                MessageBox.Show("High scores cleared!", "Done")
            Catch ex As Exception
                MessageBox.Show("Failed to clear scores: " & ex.Message)
            End Try
        End If
    End Sub
End Class

